/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.control;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.File;
import java.util.Iterator;
import java.util.List;
import java.util.prefs.Preferences;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

import rpgamevalues.data.Group;
import rpgamevalues.data.Hero;
import rpgamevalues.gui.MainFrame;
import rpgamevalues.gui.util.ExampleFileFilter;
import rpgamevalues.metadata.GameValues;
import rpgamevalues.util.Directories;



public final class GroupOperations {

  private GroupOperations() {}
  
  public static boolean addHero(JFrame parent) {
    String name = JOptionPane.showInputDialog(parent, Messages.getString("GroupOperations.EnterHeroName"), "RPGameValues", JOptionPane.QUESTION_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
    if (name == null || name.isEmpty()) return false;
    Hero hero = HeroFactory.createNewHero(name);
    Group.getGroup().addHero(hero);
    return true;
  }
  
  public static boolean removeHero(JFrame parent, String name) {
    Group.getGroup().removeHero(name);
    return true;
  }
  
  public static int autoSaveGroup(JFrame parent) {
    if (Group.getGroup().isChanged()) {
      int result = JOptionPane.showConfirmDialog(parent, Messages.getString("GroupOperations.GroupNotSaved"), "RPGameValues", JOptionPane.YES_NO_CANCEL_OPTION); //$NON-NLS-1$ //$NON-NLS-2$
      if (result == JOptionPane.CANCEL_OPTION) 
        return result;
      else if (result == JOptionPane.YES_OPTION)
      {
        return saveGroup(parent);
      }
    }    
    return JOptionPane.OK_OPTION;
  }
  
  public static int openGroup(JFrame parent) {
    JFileChooser chooser = new JFileChooser(Directories.getLastUsedDirectory(GroupOperations.class, "GroupsDir")); //$NON-NLS-1$
    chooser.setAcceptAllFileFilterUsed(true);
    chooser.setMultiSelectionEnabled(false);
    chooser.addChoosableFileFilter(new ExampleFileFilter("xml", Messages.getString("GroupOperations.Groups"))); //$NON-NLS-1$ //$NON-NLS-2$
    if (chooser.showOpenDialog(parent) == JFileChooser.APPROVE_OPTION) {
      Directories.setLastUsedDirectory(GroupOperations.class, "GroupsDir", chooser.getSelectedFile()); //$NON-NLS-1$
      return doOpenGroup(parent, chooser.getSelectedFile().getAbsolutePath());
    }
    else 
      return JOptionPane.CANCEL_OPTION;
  }
  
  public static int openGroup(JFrame parent, String fileName) {
    if (Group.getGroup().isChanged()) {
      int result = JOptionPane.showConfirmDialog(parent, Messages.getString("GroupOperations.GroupNotSaved"), "RPGameValues", JOptionPane.YES_NO_CANCEL_OPTION); //$NON-NLS-1$ //$NON-NLS-2$
      if (result == JOptionPane.CANCEL_OPTION) 
        return result;
      else if (result == JOptionPane.YES_OPTION)
      {
        result = saveGroup(parent);
        if (result == JOptionPane.CANCEL_OPTION)
          return result;
      }
    }
    return doOpenGroup(parent, fileName);
  }
  
  public static int newGroup(JFrame parent) {
    Group group = new Group();
    group.setUnchanged();
    Group.setGroup(group);
    return JOptionPane.OK_OPTION;
  }
  
  private static int doOpenGroup(JFrame parent, String fileName) {
    try {
      Group group = GroupReader.readFile(fileName);
      group.setFilePath(fileName);
      Group.setGroup(group);
      return JOptionPane.OK_OPTION;
    }
    catch (IOException e) {
      JOptionPane.showMessageDialog(parent, Messages.getString("GroupOperations.CouldNotLoadGroup") + e.getMessage(), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
      return JOptionPane.CANCEL_OPTION;
    }
  }
  
  public static int saveGroup(JFrame parent) {
    if (Group.getGroup().getFilePath().length() == 0) {
      return saveGroupAs(parent);
    }
    else {
      return doSaveGroup(parent, Group.getGroup().getFilePath());
    }
  }
  
  public static int saveGroupAs(JFrame parent) {
    JFileChooser chooser = new JFileChooser(Directories.getLastUsedDirectory(GroupOperations.class, "GroupsDir")); //$NON-NLS-1$
    chooser.setAcceptAllFileFilterUsed(true);
    chooser.setMultiSelectionEnabled(false);
    chooser.addChoosableFileFilter(new ExampleFileFilter("xml", Messages.getString("GroupOperations.Groups"))); //$NON-NLS-1$ //$NON-NLS-2$
    int result = JOptionPane.NO_OPTION;
    while (result == JOptionPane.NO_OPTION)
      if (chooser.showSaveDialog(parent) == JFileChooser.APPROVE_OPTION) {
        if (chooser.getSelectedFile().exists()) {
          result = JOptionPane.showConfirmDialog(parent, Messages.getString("GroupOperations.FileExists"), "RPGameValues", JOptionPane.YES_NO_CANCEL_OPTION); //$NON-NLS-1$ //$NON-NLS-2$
        }
        else {
          result = JOptionPane.YES_OPTION;
        }
      }
      else {
        result = JOptionPane.CANCEL_OPTION;
    }
    if (result == JOptionPane.YES_OPTION) {
      String path = chooser.getSelectedFile().getAbsolutePath();
      if (!path.endsWith(".xml")) { //$NON-NLS-1$
    	  path = path + ".xml"; //$NON-NLS-1$
      }
      Directories.setLastUsedDirectory(GroupOperations.class, "GroupsDir", new File(path)); //$NON-NLS-1$
      Group.getGroup().setFilePath(path);
      return doSaveGroup(parent, path);
    }
    else {
      return JOptionPane.CANCEL_OPTION;
    }
  }
  
  private static int doSaveGroup(JFrame parent, String filePath) {
    File tempFile = null;
    try {
      tempFile = File.createTempFile("rpgamevaluesgroup.xml", null); //$NON-NLS-1$
      File realFile = new File(filePath);
      GroupWriter.writeToFile(Group.getGroup(), tempFile.getAbsolutePath());
      Group.getGroup().setUnchanged();
      Preferences.userNodeForPackage(MainFrame.class).put("LastGroup_" + GameValues.getInstance().getName(), Group.getGroup().getFilePath()); //$NON-NLS-1$
      if (realFile.exists()) {
        @SuppressWarnings("unused")
        boolean success = realFile.delete();
        // success is actually irrelevant here
      }
      if (tempFile.renameTo(realFile) && realFile.exists()) {
        return JOptionPane.OK_OPTION;
      }
      else {
        copyFile(tempFile, realFile);
        tempFile.deleteOnExit();
        return JOptionPane.OK_OPTION;
      }
    }
    catch (IOException e) {
      JOptionPane.showMessageDialog(parent, Messages.getString("GroupOperations.CouldNotSaveGroup") //$NON-NLS-1$
          + e.getMessage(), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
      if (tempFile != null) tempFile.deleteOnExit();
    }
    return JOptionPane.CANCEL_OPTION;
  }


  private static void copyFile(File first, File second) throws IOException {
    BufferedReader in = new BufferedReader(new FileReader(first));
    try {
      PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(
          second)));
      try {
        String s = in.readLine();
        while (s != null) {
          out.println(s);
          s = in.readLine();
        }
        out.flush();
      }
      finally {
        if (out != null) out.close();
      }
    }
    finally {
      if (in != null) in.close();
    }
  }

  private static java.util.List<File> textURIListToFileList(String data) {
    java.util.List<File> list = new java.util.ArrayList<File>(1);
    for (java.util.StringTokenizer st = new java.util.StringTokenizer(data,
        "\r\n"); st.hasMoreTokens();) { //$NON-NLS-1$
      String s = st.nextToken();
      if (s.charAt(0) == '#') {
        // the line is a comment (as per the RFC 2483)
        continue;
      }
      try {
        java.net.URI uri = new java.net.URI(s);
        java.io.File file = new java.io.File(uri);
        list.add(file);
      }
      catch (java.net.URISyntaxException e) {
        // malformed URI
        continue;
      }
      catch (IllegalArgumentException e) {
        // the URI is not a valid 'file:' URI
        continue;
      }
    }
    return list;
  }

  @SuppressWarnings("unchecked")
  public static void dragEnter(DropTargetDragEvent dtde) {
    try {
      Transferable tr = dtde.getTransferable();
      List<File> fileList = null;
      DataFlavor uriListFlavor = new DataFlavor(
          "text/uri-list;class=java.lang.String"); //$NON-NLS-1$
      if (tr.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
        fileList = (java.util.List<File>) tr
            .getTransferData(DataFlavor.javaFileListFlavor);
      }
      else if (tr.isDataFlavorSupported(uriListFlavor)) {
        String data = (String) tr.getTransferData(uriListFlavor);
        fileList = textURIListToFileList(data);
      }
      if (fileList != null) {
        Iterator<File> iterator = fileList.iterator();
        while (iterator.hasNext()) {
          File f = iterator.next();
          if (f.getName().endsWith(".xml")) { //$NON-NLS-1$
            dtde.acceptDrag(DnDConstants.ACTION_COPY);
            return;
          }
        }
      }
    }
    catch (ClassNotFoundException e) {
      e.printStackTrace();
    }
    catch (UnsupportedFlavorException e) {
      e.printStackTrace();
    }
    catch (IOException e) {
      e.printStackTrace();
    }
    dtde.rejectDrag();
  }

  @SuppressWarnings("unchecked")
  public static boolean drop(DropTargetDropEvent dtde, JFrame parent) {
    boolean retCode = false;
    try {
      Transferable tr = dtde.getTransferable();
      List<File> fileList = null;
      DataFlavor uriListFlavor = new DataFlavor(
          "text/uri-list;class=java.lang.String"); //$NON-NLS-1$
      if (dtde.getDropAction() == DnDConstants.ACTION_MOVE) {
        if ((dtde.getSourceActions() & DnDConstants.ACTION_COPY) == 0) {
          dtde.rejectDrop();
          return false;
        }
      }
      else if (dtde.getDropAction() != DnDConstants.ACTION_COPY) {
        dtde.rejectDrop();
        return false;
      }
      if (tr.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
        dtde.acceptDrop(DnDConstants.ACTION_COPY);
        fileList = (java.util.List<File>) tr
            .getTransferData(DataFlavor.javaFileListFlavor);
      }
      else if (tr.isDataFlavorSupported(uriListFlavor)) {
        dtde.acceptDrop(DnDConstants.ACTION_COPY);
        String data = (String) tr.getTransferData(uriListFlavor);
        fileList = textURIListToFileList(data);
      }
      if (fileList != null) {
        Iterator<File> iterator = fileList.iterator();
        File groupFile = null;
        boolean ok = false;
        while (iterator.hasNext()) {
          File file = iterator.next();
          if (file.getName().endsWith(".xml")) { //$NON-NLS-1$
            groupFile = file;
            ok = true;
            break;
          }
        }
        if (!ok) {
          dtde.rejectDrop();
          return false;
        }
        if (groupFile != null) {
          retCode = doOpenGroup(parent, groupFile.getAbsolutePath()) == JOptionPane.OK_OPTION;
        }
        dtde.getDropTargetContext().dropComplete(true);
      }
      else {
        System.err.println("Rejected"); //$NON-NLS-1$
        dtde.rejectDrop();
      }
    }
    catch (IOException io) {
      io.printStackTrace();
      dtde.rejectDrop();
    }
    catch (ClassNotFoundException e) {
      e.printStackTrace();
      dtde.rejectDrop();
    }
    catch (UnsupportedFlavorException ufe) {
      ufe.printStackTrace();
      dtde.rejectDrop();
    }
    return retCode;
  }
 }
