/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.control;

import java.io.IOException;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import rpgamevalues.data.Group;
import rpgamevalues.data.Hero;
import rpgamevalues.metadata.DiceSpecification;
import rpgamevalues.metadata.GameValues;
import rpgamevalues.metadata.IGameValue;


final class GroupReader {

  private GroupReader() {}
  
  public static Group readFile(String fileName) throws IOException {
    try {
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
      DocumentBuilder db = dbf.newDocumentBuilder();
      Document dom = db.parse(fileName);

      Element docEle = dom.getDocumentElement();
      
      if (docEle == null) {
        throw new IOException(Messages.getString("GroupReader.GroupElementMissing")); //$NON-NLS-1$
      }
      
      return readGroup(docEle);
    }
    catch(ParserConfigurationException pce) {
      throw new IOException(pce.getLocalizedMessage(), pce);
    }
    catch(SAXException se) {
      throw new IOException(se.getLocalizedMessage(), se);
    }
    catch (IOException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
  }
  
  private static Group readGroup(Element el) throws IOException {
    Group group = new Group();
    NodeList heroes = el.getElementsByTagName("hero"); //$NON-NLS-1$
    for (int i = 0 ; i < heroes.getLength(); ++i) {
      Hero hero = readHero((Element)heroes.item(i));
      if (group.getHero(hero.getName()) != null) {
        throw new IOException(Messages.getString("GroupReader.DuplicateHeroName") + hero.getName()); //$NON-NLS-1$
      }
      group.addHero(hero);
    }
    group.setUnchanged();
    return group;
  }
  
  private static Hero readHero(Element el) throws IOException {
    String heroName = el.getAttribute("name"); //$NON-NLS-1$
    if (heroName == null) {
      throw new IOException(Messages.getString("GroupReader.HeroMustHaveName")); //$NON-NLS-1$
    }
    Hero hero = HeroFactory.createNewHero(heroName);
    String selectedGroup = el.getAttribute("selectedGroup"); //$NON-NLS-1$
    if (selectedGroup != null && selectedGroup.length() > 0) {
      hero.setCurrentGroup(parseInt(selectedGroup));
    }
    else {
      hero.setCurrentGroup(0);
    }
    NodeList nl = el.getElementsByTagName("valuegroup"); //$NON-NLS-1$
    HashMap<String, Element> valueGroups = new HashMap<String, Element>();
    for (int i = 0; i < nl.getLength(); ++i) {
      Element groupElement = (Element)nl.item(i);
      String groupName = groupElement.getAttribute("name"); //$NON-NLS-1$
      if (groupName == null) {
        throw new IOException(Messages.getString("GroupReader.ValueGroupMustHaveName")); //$NON-NLS-1$
      }
      valueGroups.put(groupName, groupElement);
    }
    for (String valueGroup : GameValues.getInstance().getGroups()) {
      if (valueGroups.containsKey(valueGroup)) {
        readValueGroup(hero, valueGroup, valueGroups.get(valueGroup));
      }
    }
    return hero;
  }
  
  private static void readValueGroup(Hero hero, String valueGroup, Element el) throws IOException {
    if (GameValues.getInstance().hasConfigurations(valueGroup)) {
      NodeList nl = el.getElementsByTagName("configurations"); //$NON-NLS-1$
      if (nl.getLength() > 1) {
        throw new IOException(Messages.getString("GroupReader.OnlyOneConfigurationsElementAllowed")); //$NON-NLS-1$
      }
      else if (nl.getLength() == 1) {
        Element configurationsElement = (Element)nl.item(0);
        String currentConfig = configurationsElement.getAttribute("current"); //$NON-NLS-1$
        if (currentConfig == null) {
          throw new IOException(Messages.getString("GroupReader.CurrentConfigurationNameMissing")); //$NON-NLS-1$
        }
        NodeList configNodes = configurationsElement.getElementsByTagName("configuration"); //$NON-NLS-1$
        if (configNodes.getLength() == 0) {
          throw new IOException(Messages.getString("GroupReader.ConfigurationMissing")); //$NON-NLS-1$
        }
        for (int i = 0; i < configNodes.getLength(); ++i) {
          Element configElement = (Element)configNodes.item(i);
          String configName = configElement.getAttribute("name"); //$NON-NLS-1$
          if (configName == null) {
            throw new IOException(Messages.getString("GroupReader.ConfigurationNameMissing")); //$NON-NLS-1$
          }
          if (i == 0) {
            hero.changeCurrentConfigurationName(valueGroup, configName);
          }
          else {
            hero.addConfiguration(valueGroup, configName);
            hero.setCurrentConfiguration(valueGroup, configName);
          }
          readConfigurationValues(hero, valueGroup, configElement);
        }
        hero.setCurrentConfiguration(valueGroup, currentConfig);
      }
      else {
        readConfigurationValues(hero, valueGroup, el);
      }
    }
    else {
      readConfigurationValues(hero, valueGroup, el);
    }
  }
  
  private static void readConfigurationValues(Hero hero, String valueGroup, Element parent) throws IOException {
    NodeList nodes = parent.getElementsByTagName("value"); //$NON-NLS-1$
    HashMap<String, Element> values = new HashMap<String, Element>();
    for (int i = 0; i < nodes.getLength(); ++i) {
      Element value = (Element)nodes.item(i);
      String name = value.getAttribute("name"); //$NON-NLS-1$
      if (name == null) {
        throw new IOException(Messages.getString("GroupReader.ValueNameMissing")); //$NON-NLS-1$
      }
      values.put(name, value);
    }
    for (IGameValue gameValue : GameValues.getInstance().getGameValues(valueGroup)) {
      if (values.containsKey(gameValue.getName())) {
        Element value = values.get(gameValue.getName());
        if (!gameValue.isCalculated()) {
          String current = value.getAttribute("current"); //$NON-NLS-1$
          if (current == null) {
            throw new IOException(Messages.getString("GroupReader.CurrentValueMissing") + gameValue.getName()); //$NON-NLS-1$
          }
          switch (gameValue.getGameValueType()) {
          case DiceSpecification:
            hero.setCurrentValue(valueGroup, gameValue.getName(), parseDS(current));
            break;
          case Boolean:
            hero.setCurrentValue(valueGroup, gameValue.getName(), parseBoolean(current));
            break;
          case String:
        	hero.setCurrentValue(valueGroup, gameValue.getName(), current);
        	break;
          default:
            hero.setCurrentValue(valueGroup, gameValue.getName(), parseInt(current));
            break;
          }
          if (gameValue.hasDefault()) {
            String defaultValue = value.getAttribute("default"); //$NON-NLS-1$
            if (defaultValue != null && defaultValue.length() > 0) {
              switch (gameValue.getGameValueType()) {
              case DiceSpecification:
                hero.setDefaultValue(valueGroup, gameValue.getName(), parseDS(defaultValue));
                break;
              case Boolean:
                hero.setDefaultValue(valueGroup, gameValue.getName(), parseBoolean(defaultValue));
              default:
                hero.setDefaultValue(valueGroup, gameValue.getName(), parseInt(defaultValue));
                break;
              }
            }
            else {
              hero.setDefaultValue(valueGroup, gameValue.getName(), hero.getCurrentValue(valueGroup, gameValue.getName()));
            }
          }
        }
        String visible = value.getAttribute("visible"); //$NON-NLS-1$
        hero.setValueShown(valueGroup, gameValue.getName(), !"false".equals(visible)); //$NON-NLS-1$
      }
    }
  }
  
  private static Integer parseInt(String value) throws IOException {
    try {
      return Integer.parseInt(value);
    }
    catch (NumberFormatException e) {
      throw new IOException(Messages.getString("GroupReader.ValueMustBeNumber") + value, e); //$NON-NLS-1$
    }
  }
  
  private static DiceSpecification parseDS(String value) throws IOException {
    if (value.equals("null")) { //$NON-NLS-1$
      return null;
    }
    try {
      return DiceSpecification.parse(value);
    }
    catch (NumberFormatException e) {
      throw new IOException(Messages.getString("GroupReader.ValueMustBeDiceSpecification") + value, e); //$NON-NLS-1$
    }
  }
  
  private static Boolean parseBoolean(String value) throws IOException {
    return Boolean.parseBoolean(value);
  }
}
