/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.control;

import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import rpgamevalues.data.Group;
import rpgamevalues.data.Hero;
import rpgamevalues.metadata.GameValues;
import rpgamevalues.metadata.IGameValue;


final class GroupWriter {

  private GroupWriter() {}
  
  public static void writeToFile(Group group, String fileName) throws IOException {
    FileOutputStream file = new FileOutputStream(fileName);
    PrintWriter out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(file, "UTF-8"))); //$NON-NLS-1$
    StreamResult streamResult = new StreamResult(out);
    
    try {
      SAXTransformerFactory tf = (SAXTransformerFactory)SAXTransformerFactory.newInstance();
      TransformerHandler hd = tf.newTransformerHandler();
      Transformer serializer = hd.getTransformer();
      serializer.setOutputProperty(OutputKeys.ENCODING, "UTF-8"); //$NON-NLS-1$
      serializer.setOutputProperty(OutputKeys.INDENT, "yes"); //$NON-NLS-1$
      hd.setResult(streamResult);
      
      writeToDocument(hd, group);
      
    }
    catch (TransformerConfigurationException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (SAXException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    
    out.flush();
    out.close();
  }
  
  private static void writeToDocument(TransformerHandler hd, Group group) throws SAXException {
    AttributesImpl atts = new AttributesImpl();
    hd.startDocument();
    hd.startElement("", "", "group", atts); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    for (String hero : group.getHeroes()) {
      writeHero(hd, group.getHero(hero));
    }
    hd.endElement("", "", "group"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    hd.endDocument();
  }
  
  private static void writeHero(TransformerHandler hd, Hero hero) throws SAXException {
    AttributesImpl atts = new AttributesImpl();
    atts.addAttribute("", "", "name", "CDATA", hero.getName()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
    atts.addAttribute("", "", "selectedGroup", "CDATA", "" + hero.getCurrentGroup()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
    hd.startElement("", "", "hero", atts); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    atts.clear();
    for (String valueGroup : GameValues.getInstance().getGroups()) {
      atts.addAttribute("", "", "name", "CDATA", valueGroup); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      hd.startElement("", "", "valuegroup", atts); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      atts.clear();
      if (GameValues.getInstance().hasConfigurations(valueGroup)) {
        String currentConfiguration = hero.getCurrentConfiguration(valueGroup);
        try {
          atts.addAttribute("", "", "current", "CDATA", currentConfiguration); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
          hd.startElement("", "", "configurations", atts); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
          atts.clear();
          for (String configuration : hero.getConfigurations(valueGroup)) {
            atts.addAttribute("", "", "name", "CDATA", configuration); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
            hd.startElement("", "", "configuration", atts); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            atts.clear();
            hero.setCurrentConfiguration(valueGroup, configuration);
            writeConfigurationValues(hd, hero, valueGroup);
            hd.endElement("", "", "configuration"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
          }
          hd.endElement("", "", "configurations"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        }
        finally {
          hero.setCurrentConfiguration(valueGroup, currentConfiguration);
        }
      }
      else {
        writeConfigurationValues(hd, hero, valueGroup);
      }
      hd.endElement("", "", "valuegroup"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    }
    hd.endElement("", "", "hero"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
  }
  
  private static void writeConfigurationValues(TransformerHandler hd, Hero hero, String valueGroup) throws SAXException {
    for (IGameValue gameValue : GameValues.getInstance().getGameValues(valueGroup)) {
      AttributesImpl atts = new AttributesImpl();
      atts.addAttribute("", "", "name", "CDATA", gameValue.getName()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
      if (!gameValue.isCalculated()) {
        atts.addAttribute("", "", "current", "CDATA", "" + ValueCalculation.getCurrentValue(valueGroup, gameValue, hero)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
        if (gameValue.hasDefault()) {
          atts.addAttribute("", "", "default", "CDATA", "" + ValueCalculation.getDefaultValue(valueGroup, gameValue, hero)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
        }
      }
      atts.addAttribute("", "", "visible", "CDATA",  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
          hero.isValueShown(valueGroup, gameValue.getName(), gameValue.isVisibleByDefault()) ? "true" : "false"); //$NON-NLS-1$ //$NON-NLS-2$
      hd.startElement("", "", "value", atts); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      hd.endElement("", "", "value"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
    }
  }
}
