/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.data;

import java.util.HashMap;

final class Configurations {

  private HashMap<String, GameValueCollection> mConfigurations = new HashMap<String, GameValueCollection>();
  
  private boolean changed = false;
  
  private String mCurrentConfiguration = Messages.getString("Configurations.Default"); //$NON-NLS-1$
  
  public Configurations() {
    addConfiguration(Messages.getString("Configurations.Default")); //$NON-NLS-1$
  }
  
  public String[] getConfigurations() {
    String[] result = new String[mConfigurations.keySet().size()];
    mConfigurations.keySet().toArray(result);
    return result;
  }
  
  public void addConfiguration(String name) {
    if (!mConfigurations.containsKey(name)) {
      mConfigurations.put(name, new GameValueCollection());
      changed = true;
    }
  }
  
  public void removeConfiguration(String name) {
    if (mConfigurations.containsKey(name) && mConfigurations.size() > 1) {
      mConfigurations.remove(name);
      changed = true;
    }
  }
  
  public void duplicateConfiguration(String newName) {
    if (!mConfigurations.containsKey(newName)) {
      mConfigurations.put(newName, mConfigurations.get(mCurrentConfiguration).clone());
      changed = true;
    }
  }
  
  public <T> T getDefaultValue(String name) {
    return mConfigurations.get(mCurrentConfiguration).<T>getDefaultValue(name);
  }
  
  public <T> T getCurrentValue(String name) {
    return mConfigurations.get(mCurrentConfiguration).<T>getCurrentValue(name);
  }
  
  public <T> void setDefaultValue(String name, T value) {
    mConfigurations.get(mCurrentConfiguration).setDefaultValue(name, value);
  }
  
  public <T> void setCurrentValue(String name, T value) {
    mConfigurations.get(mCurrentConfiguration).setCurrentValue(name, value);
  }
  
  public boolean isValueShown(String name, boolean defaultValue) {
    return mConfigurations.get(mCurrentConfiguration).isValueShown(name, defaultValue);
  }
  
  public void setValueShown(String name, boolean shown) {
    mConfigurations.get(mCurrentConfiguration).setValueShown(name, shown);
  }
  
  public String getCurrentConfiguration() {
    return mCurrentConfiguration;
  }
  
  public void setCurrentConfiguration(String name) {
    if (!mCurrentConfiguration.equals(name)) {
      mCurrentConfiguration = name;
      changed = true;
    }
  }
  
  public void changeCurrentConfigurationName(String name) {
    if (!mCurrentConfiguration.equals(name)) {
      GameValueCollection collection = mConfigurations.get(mCurrentConfiguration);
      mConfigurations.remove(mCurrentConfiguration);
      mConfigurations.put(name, collection);
      mCurrentConfiguration = name;
      changed = true;
    }
  }
  
  public boolean isChanged() {
    for (GameValueCollection entry : mConfigurations.values()) {
      if (entry.isChanged()) {
        return true;
      }
    }
    return changed;
  }
  
  public void setUnchanged() {
    changed = false;
    for (GameValueCollection entry : mConfigurations.values()) {
      entry.setUnchanged();
    }
  }

}
