/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.data;

import java.util.HashMap;

final class GameValueCollection implements Cloneable {

  private HashMap<String, GameValue<?>> mValues = new HashMap<String, GameValue<?>>();
  
  private HashMap<String, Boolean> mShownValues = new HashMap<String, Boolean>();
  
  private static <T> boolean equal(T t1, T t2) {
    if (t1 == null) {
      return t2 == null;
    }
    else {
      if (t2 == null)
        return false;
      else 
        return t1.equals(t2);
    }
  }
  
  @SuppressWarnings("unchecked")
  public <T> void setDefaultValue(String name, T value) {
    if (!equal(getDefaultValue(name), value)) {
      if (!mValues.containsKey(name)) {
        mValues.put(name, new GameValue<T>());
      }
      changed = true;
      ((GameValue<T>)mValues.get(name)).setDefaultValue(value);
    }
  }
  
  @SuppressWarnings("unchecked")
  public <T> void setCurrentValue(String name, T value) {
    if (!equal(getCurrentValue(name), value)) {
      if (!mValues.containsKey(name)) {
        mValues.put(name, new GameValue<T>());
      }
      changed = true;
      ((GameValue<T>)mValues.get(name)).setCurrentValue(value);
    }
  }
  
  @SuppressWarnings("unchecked")
  public <T> T getCurrentValue(String name) {
    if (!mValues.containsKey(name)) {
      return null;
    }
    return ((GameValue<T>)mValues.get(name)).getCurrentValue();
  }
  
  @SuppressWarnings("unchecked")
  public <T> T getDefaultValue(String name) {
    if (!mValues.containsKey(name)) {
      return null;
    }
    return ((GameValue<T>)mValues.get(name)).getDefaultValue();
  }
  
  public boolean isValueShown(String name, boolean defaultValue) {
    if (mShownValues.containsKey(name)) {
      return mShownValues.get(name);
    }
    else {
      return defaultValue;
    }
  }
  
  public void setValueShown(String name, boolean shown) {
    if (!mShownValues.containsKey(name) || mShownValues.get(name) != shown) {
      mShownValues.put(name, shown);
      changed = true;
    }
  }
  
  public GameValueCollection clone() {
    try {
      GameValueCollection clone = (GameValueCollection)super.clone();
      clone.mValues = new HashMap<String, GameValue<?>>();
      for (String name : mValues.keySet()) {
        clone.mValues.put(name, mValues.get(name).clone());
      }
      clone.mShownValues = new HashMap<String, Boolean>();
      for (String name : mShownValues.keySet()) {
        clone.mShownValues.put(name, mShownValues.get(name));
      }
      clone.changed = true;
      return clone;
    }
    catch (CloneNotSupportedException e) {
      throw new InternalError();
    }
  }
  
  private boolean changed = false;
  
  public boolean isChanged() {
    return changed;
  }
  
  public void setUnchanged() {
    changed = false;
  }
}
