/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import rpgamevalues.data.Hero;
import rpgamevalues.metadata.GameValues;
import rpgamevalues.metadata.IGameValue;


class ConfigurationsPanel extends HeroChangerPanel {

  private String groupName;
  private Hero hero;
  private JComboBox configurationsBox;
  private JButton configurationsButton;
  private ValueGroupPanel valuesPanel;
  
  public ConfigurationsPanel(IHeroFrame frame, Hero hero, String groupName) {
    super(frame);
    this.hero = hero;
    this.groupName = groupName;
    setLayout(new BorderLayout());
    add(getConfigurationsPanel(), BorderLayout.NORTH);
    valuesPanel = new ValueGroupPanel(frame, hero, groupName);
    add(valuesPanel, BorderLayout.CENTER);
  }
  
  private JPanel getConfigurationsPanel() {
    JPanel configurationsPanel = new JPanel();
    configurationsPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 0));
    configurationsPanel.setLayout(new BoxLayout(configurationsPanel, BoxLayout.LINE_AXIS));
    configurationsPanel.add(getConfigurationsBox());
    configurationsPanel.add(Box.createHorizontalStrut(5));
    configurationsPanel.add(getConfigurationsButton());
    configurationsPanel.add(Box.createHorizontalGlue());
    return configurationsPanel;
  }
  
  private boolean listenForConfigurations = false;
  
  /**
   * @return
   */
  private Component getConfigurationsButton() {
    if (configurationsButton == null) {
      configurationsButton = new JButton("..."); //$NON-NLS-1$
      configurationsButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          showConfigurationsMenu();
        }
      });
    }
    return configurationsButton;
  }

  /**
   * 
   */
  protected void showConfigurationsMenu() {
    JMenu menu = new JMenu();
    JMenuItem copyItem = new JMenuItem(Messages.getString("ConfigurationsPanel.Copy")); //$NON-NLS-1$
    copyItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        copyConfiguration();
      }
    });
    menu.add(copyItem);
    JMenuItem renameItem = new JMenuItem(Messages.getString("ConfigurationsPanel.Rename")); //$NON-NLS-1$
    renameItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        renameConfiguration();
      }
    });
    menu.add(renameItem);
    JMenuItem deleteItem = new JMenuItem(Messages.getString("ConfigurationsPanel.Delete")); //$NON-NLS-1$
    deleteItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        deleteConfiguration();
      }
    });
    menu.add(deleteItem);
    JMenu shownValuesMenu = new JMenu(Messages.getString("ConfigurationsPanel.Show")); //$NON-NLS-1$
    for (final IGameValue gameValue : GameValues.getInstance().getGameValues(groupName)) {
      final JMenuItem showValuesItem = new JCheckBoxMenuItem(gameValue.getName());
      showValuesItem.setSelected(hero.isValueShown(groupName, gameValue.getName(), gameValue.isVisibleByDefault()));
      showValuesItem.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          toggleShowValue(gameValue.getName(), showValuesItem.isSelected());
        }
      });
      shownValuesMenu.add(showValuesItem);
    }
    menu.addSeparator();
    menu.add(shownValuesMenu);
    menu.getPopupMenu().show(this, getConfigurationsButton().getX(), getConfigurationsButton().getY() + getConfigurationsButton().getHeight());
  }
  
  private boolean configurationExists(String name) {
    for (String config : hero.getConfigurations(groupName)) {
      if (config.equals(name))
        return true;
    }
    return false;
  }
  
  private void copyConfiguration() {
    String newName = JOptionPane.showInputDialog(this, Messages.getString("ConfigurationsPanel.EnterConfigurationName"), "RPGameValues", JOptionPane.QUESTION_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
    if (newName == null)
      return;
    if ("".equals(newName)) { //$NON-NLS-1$
      JOptionPane.showMessageDialog(this, Messages.getString("ConfigurationsPanel.ConfigurationMustHaveName"), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
      return;
    }
    if (configurationExists(newName)) {
      JOptionPane.showMessageDialog(this, Messages.getString("ConfigurationsPanel.ConfigurationExists"), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
      return;
    }
    hero.duplicateConfiguration(groupName, newName);
    hero.setCurrentConfiguration(groupName, newName);
    fillConfigurationsBox();
    fireHeroChanged();
  }
  
  private void renameConfiguration() {
    String newName = JOptionPane.showInputDialog(this, Messages.getString("ConfigurationsPanel.EnterNewConfigurationName"), "RPGameValues", JOptionPane.QUESTION_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
    if (newName == null)
      return;
    if ("".equals(newName)) { //$NON-NLS-1$
      JOptionPane.showMessageDialog(this, Messages.getString("ConfigurationsPanel.ConfigurationMustHaveName"), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
      return;
    }
    if (newName.equals(hero.getCurrentConfiguration(groupName))) {
      return;
    }
    if (configurationExists(newName)) {
      JOptionPane.showMessageDialog(this, Messages.getString("ConfigurationsPanel.ConfigurationExists"), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
      return;
    }
    hero.changeCurrentConfigurationName(groupName, newName);
    fillConfigurationsBox();
    // no change notification necessary, no other values change
  }
  
  private void deleteConfiguration() {
    if (hero.getConfigurations(groupName).length < 2) {
      JOptionPane.showMessageDialog(this, Messages.getString("ConfigurationsPanel.MustHaveAtLeastOneConfiguration"), "RPGameValues", JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
      return;
    }
    if (JOptionPane.showConfirmDialog(this, Messages.getString("ConfigurationsPanel.ReallyDeleteConfiguration"), "RPGameValues", JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) { //$NON-NLS-1$ //$NON-NLS-2$
      hero.removeConfiguration(groupName, hero.getCurrentConfiguration(groupName));
      hero.setCurrentConfiguration(groupName, hero.getConfigurations(groupName)[0]);
      fillConfigurationsBox();
      fireHeroChanged();
    }
  }
  
  private void toggleShowValue(String valueName, boolean defaultValue) {
    hero.setValueShown(groupName, valueName, !hero.isValueShown(groupName, valueName, defaultValue));
    valuesPanel.shownValuesToggled();
  }
  
  private void fillConfigurationsBox() {
    listenForConfigurations = false;
    configurationsBox.removeAllItems();
    for (String configuration : hero.getConfigurations(groupName)) {
      configurationsBox.addItem(configuration);
    }
    configurationsBox.setSelectedItem(hero.getCurrentConfiguration(groupName));    
    listenForConfigurations = true;
  }

  private JComboBox getConfigurationsBox() {
    if (configurationsBox == null) {
      configurationsBox = new JComboBox();
      fillConfigurationsBox();
      configurationsBox.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          configurationChanged();
        }
      });
    }
    return configurationsBox;
  }
  
  private void configurationChanged() {
    if (!listenForConfigurations) return;
    hero.setCurrentConfiguration(groupName, getConfigurationsBox().getSelectedItem().toString());
    valuesPanel.shownValuesToggled();
  }
  
  public void heroChanged() {
    // nothing
  }
}
