/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */package rpgamevalues.gui;

import javax.swing.JOptionPane;


import java.awt.Dimension;
import java.awt.Rectangle;

import javax.swing.JPanel;
import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;
import java.util.prefs.Preferences;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.BorderFactory;
import javax.swing.DefaultListModel;
import javax.swing.ImageIcon;
import javax.swing.JScrollPane;
import javax.swing.JList;
import javax.swing.JButton;

import rpgamevalues.control.GroupOperations;
import rpgamevalues.control.Version;
import rpgamevalues.data.Group;
import rpgamevalues.metadata.GameValues;

public final class MainFrame extends FrameController {

  private JPanel jContentPane = null;
  private JPanel toolBarPanel = null;
  private JPanel groupButtonsPanel = null;
  private JScrollPane jScrollPane = null;
  private JList heroList = null;
  private JButton addButton = null;
  private JButton removeButton = null;
  private JButton editButton = null;
  private JButton newButton = null;
  private JButton openButton = null;
  private JButton saveButton = null;
  private JButton saveAsButton = null;
  private JButton settingsButton = null;
  private JButton aboutButton = null;

  public MainFrame() {
    super("RPGameValues - " + GameValues.getInstance().getName()); //$NON-NLS-1$
		initialize();
  }
  
  /**
   * This method initializes this
   * 
   */
  private void initialize() {
        this.setContentPane(getJContentPane());
        this.setTitle("RPGameValues - " + GameValues.getInstance().getName()); //$NON-NLS-1$
        this.pack();
        this.addWindowListener(new WindowAdapter() {

          @Override
          public void windowClosed(WindowEvent arg0) {
          }

          @Override
          public void windowClosing(WindowEvent arg0) {
            saveAndExit();
          }

          public void windowIconified(WindowEvent e) {
            FrameManagement.getInstance().iconifyAllFrames(MainFrame.this);
          }

          public void windowDeiconified(WindowEvent e) {
            FrameManagement.getInstance().deiconifyAllFrames(MainFrame.this);
          }

        });
        this.setDefaultCloseOperation(javax.swing.JFrame.DO_NOTHING_ON_CLOSE);
  }
  
  private void saveAndExit() {
    if (GroupOperations.autoSaveGroup(this) != JOptionPane.OK_OPTION) return;
    storeLayout();
    FrameManagement.getInstance().setExiting();
    FrameManagement.getInstance().closeAllFrames(this);
    System.exit(0);
  }

  private void storeLayout() {
    if (Group.getGroup().getFilePath() != null && !"".equals(Group.getGroup().getFilePath())) { //$NON-NLS-1$
      File file = new File(Group.getGroup().getFilePath());
      String name = file.getName();
      File layoutsFile = new File(rpgamevalues.util.Directories.getUserDataPath() +  name + ".layout"); //$NON-NLS-1$
      try {
        FrameLayouts.getInstance().storeToFile(layoutsFile.getAbsolutePath());
      }
      catch (IOException e) {
        JOptionPane.showMessageDialog(this, Messages.getString("MainFrame.CouldNotSaveLayouts")  //$NON-NLS-1$
            + e.getMessage(), Messages.getString("MainFrame.Error"), JOptionPane.ERROR_MESSAGE);       //$NON-NLS-1$
      }    
    }
  }

  /* (non-Javadoc)
   * @see rpgamevalues.gui.SubFrame#getHierarchyPosition()
   */
  @Override
  int getHierarchyPosition() {
    return 0;
  }

  public void openFile(String string) {
    if (GroupOperations.openGroup(this, string) != JOptionPane.CANCEL_OPTION) {
      groupOpened();
    }
  }
  
  private static final int SPACE = 5;

  /**
   * This method initializes jContentPane	
   * 	
   * @return javax.swing.JPanel	
   */
  private JPanel getJContentPane() {
    if (jContentPane == null) {
      jContentPane = new JPanel();
      jContentPane.setLayout(new BorderLayout());
      jContentPane.add(getToolBarPanel(), BorderLayout.NORTH);
      jContentPane.add(getGroupButtonsPanel(), BorderLayout.EAST);
      jContentPane.add(getJScrollPane(), BorderLayout.CENTER);
    }
    return jContentPane;
  }

  /**
   * This method initializes toolBarPanel	
   * 	
   * @return javax.swing.JPanel	
   */
  private JPanel getToolBarPanel() {
    if (toolBarPanel == null) {
      toolBarPanel = new JPanel();
      toolBarPanel.setLayout(new BoxLayout(getToolBarPanel(), BoxLayout.X_AXIS));
      toolBarPanel.setBorder(BorderFactory.createEmptyBorder(0, 5, 5, 5));
      toolBarPanel.add(getNewButton(), null);
      toolBarPanel.add(Box.createHorizontalStrut(SPACE));
      toolBarPanel.add(getOpenButton(), null);
      toolBarPanel.add(Box.createHorizontalStrut(SPACE));
      toolBarPanel.add(getSaveButton(), null);
      toolBarPanel.add(Box.createHorizontalStrut(SPACE));
      toolBarPanel.add(getSaveAsButton(), null);
      toolBarPanel.add(Box.createHorizontalStrut(SPACE));
      toolBarPanel.add(getSettingsButton(), null);
      toolBarPanel.add(Box.createHorizontalStrut(SPACE));
      toolBarPanel.add(getAboutButton(), null);
      toolBarPanel.add(Box.createHorizontalStrut(SPACE));
      toolBarPanel.add(Box.createHorizontalGlue());
    }
    return toolBarPanel;
  }

  /**
   * This method initializes groupButtonsPanel	
   * 	
   * @return javax.swing.JPanel	
   */
  private JPanel getGroupButtonsPanel() {
    if (groupButtonsPanel == null) {
      groupButtonsPanel = new JPanel();
      groupButtonsPanel.setLayout(new BoxLayout(getGroupButtonsPanel(), BoxLayout.Y_AXIS));
      groupButtonsPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
      groupButtonsPanel.add(getAddButton(), null);
      groupButtonsPanel.add(Box.createVerticalStrut(SPACE));
      groupButtonsPanel.add(getRemoveButton(), null);
      groupButtonsPanel.add(Box.createVerticalStrut(SPACE));
      groupButtonsPanel.add(getEditButton(), null);
      groupButtonsPanel.add(Box.createVerticalGlue());
    }
    return groupButtonsPanel;
  }

  /**
   * This method initializes jScrollPane	
   * 	
   * @return javax.swing.JScrollPane	
   */
  private JScrollPane getJScrollPane() {
    if (jScrollPane == null) {
      jScrollPane = new JScrollPane();
      jScrollPane.setViewportView(getHeroList());
      jScrollPane.setBorder(BorderFactory.createCompoundBorder(
          BorderFactory.createEmptyBorder(SPACE, SPACE, SPACE, SPACE),
          BorderFactory.createEtchedBorder()));
    }
    return jScrollPane;
  }

  /**
   * This method initializes heroList	
   * 	
   * @return javax.swing.JList	
   */
  private JList getHeroList() {
    if (heroList == null) {
      heroList = new JList();
      heroList.setPreferredSize(new Dimension(250, 100));
      heroList.setModel(new DefaultListModel());
      heroList.addMouseListener(new MouseAdapter() {
        public void mouseClicked(MouseEvent e) {
          if (e.getClickCount() > 1) {
            openHeroes();
          }
        }
      });
    }
    return heroList;
  }

  /**
   * This method initializes addButton	
   * 	
   * @return javax.swing.JButton	
   */
  private JButton getAddButton() {
    if (addButton == null) {
      addButton = new JButton();
      addButton.setIcon(new ImageIcon(getClass().getResource("increase.png"))); //$NON-NLS-1$
      addButton.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent e) {
          if (GroupOperations.addHero(MainFrame.this)) {
            refillHeroList();
          }
        }
      });
    }
    return addButton;
  }

  /**
   * This method initializes removeButton	
   * 	
   * @return javax.swing.JButton	
   */
  private JButton getRemoveButton() {
    if (removeButton == null) {
      removeButton = new JButton();
      removeButton.setIcon(new ImageIcon(getClass().getResource("decrease_enabled.png"))); //$NON-NLS-1$
      removeButton.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent e) {
          removeSelectedHeroes();
        }
      });
    }
    return removeButton;
  }
  
  private void removeSelectedHeroes() {
    for (int index : heroList.getSelectedIndices()) {
      String name = heroList.getModel().getElementAt(index).toString();
      GroupOperations.removeHero(this, name);
    }
    refillHeroList();
  }

  private void openHeroes() {
    for (int index : heroList.getSelectedIndices()) {
      String name = heroList.getModel().getElementAt(index).toString();
      SubFrame existing = FrameManagement.getInstance().getFrame(name);
      if (existing == null) {
        HeroFrame frame = new HeroFrame(Group.getGroup().getHero(name));
        frame.setVisible(true);
      }
      else {
        Preferences prefs = Preferences.userNodeForPackage(FrameManagement.class);    
        if (prefs.getBoolean("BringWindowsToTop", true)) { //$NON-NLS-1$
          existing.toFront();
        }
      }
    }
  }

  /**
   * This method initializes editButton	
   * 	
   * @return javax.swing.JButton	
   */
  private JButton getEditButton() {
    if (editButton == null) {
      editButton = new JButton();
      editButton.setIcon(new ImageIcon(getClass().getResource("edit.png"))); //$NON-NLS-1$
      editButton.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent e) {
          openHeroes();
        }
      });
    }
    return editButton;
  }
  
  private JButton getNewButton() {
    if (newButton == null) {
      newButton = new JButton();
      newButton.setIcon(new ImageIcon(getClass().getResource("New24.gif"))); //$NON-NLS-1$
      newButton.setToolTipText(Messages.getString("MainFrame.New")); //$NON-NLS-1$
      newButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          newGroup();
        }
      });
    }
    return newButton;
  }
  
  private void newGroup() {
	int result = GroupOperations.autoSaveGroup(this);
	if (result == JOptionPane.CANCEL_OPTION) {
		return;
	}
	else if (result == JOptionPane.OK_OPTION) {
      storeLayout();
    }
    if (GroupOperations.newGroup(this) != JOptionPane.CANCEL_OPTION) {
      groupOpened();
    }
  }

  /**
   * This method initializes openButton	
   * 	
   * @return javax.swing.JButton	
   */
  private JButton getOpenButton() {
    if (openButton == null) {
      openButton = new JButton();
      openButton.setIcon(new ImageIcon(getClass().getResource("Open24.gif"))); //$NON-NLS-1$
      openButton.setToolTipText(Messages.getString("MainFrame.Open")); //$NON-NLS-1$
      openButton.addActionListener(new ActionListener()  {
        public void actionPerformed(ActionEvent e) {
          openGroup();
        }
      });
    }
    return openButton;
  }
  
  private void openGroup() {
	int result = GroupOperations.autoSaveGroup(this);
	if (result == JOptionPane.CANCEL_OPTION) {
		return;
	}
	else if (result == JOptionPane.OK_OPTION) {
      storeLayout();
    }
    if (GroupOperations.openGroup(this) != JOptionPane.CANCEL_OPTION) {
      groupOpened();
    }
  }
  
  private void groupOpened() {
    FrameManagement.getInstance().closeAllFrames(this);
    refillHeroList();
    if (Group.getGroup().getFilePath() != null && !"".equals(Group.getGroup().getFilePath())) { //$NON-NLS-1$
      File groupFile = new File(Group.getGroup().getFilePath());
      File layoutsFile = new File(rpgamevalues.util.Directories.getUserDataPath() + groupFile.getName() + ".layout"); //$NON-NLS-1$
      if (layoutsFile.exists()) try {
        FrameLayouts.getInstance().readFromFile(layoutsFile.getAbsolutePath());
        FrameLayouts.getInstance().restoreLastLayout();
      }
      catch (IOException e) {
        JOptionPane.showMessageDialog(this, Messages.getString("MainFrame.CouldNotLoadLayouts")  //$NON-NLS-1$
            + e.getMessage(), Messages.getString("MainFrame.Error"), JOptionPane.ERROR_MESSAGE);               //$NON-NLS-1$
      }

      Preferences.userNodeForPackage(MainFrame.class).put("LastGroup_" + GameValues.getInstance().getName(), Group.getGroup().getFilePath()); //$NON-NLS-1$
    }
    
  }
  
  private void refillHeroList() {
    DefaultListModel model = (DefaultListModel)heroList.getModel();
    model.clear();
    for (String name : Group.getGroup().getHeroes()) {
      model.addElement(name);
    }    
    getRemoveButton().setEnabled(Group.getGroup().getHeroes().length > 0);
  }

  /**
   * This method initializes saveButton	
   * 	
   * @return javax.swing.JButton	
   */
  private JButton getSaveButton() {
    if (saveButton == null) {
      saveButton = new JButton();
      saveButton.setIcon(new ImageIcon(getClass().getResource("Save24.gif"))); //$NON-NLS-1$
      saveButton.setToolTipText(Messages.getString("MainFrame.Save")); //$NON-NLS-1$
      saveButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          saveGroup();
        }
      });
    }
    return saveButton;
  }
  
  private void saveGroup() {
    GroupOperations.saveGroup(this);
  }

  private JButton getSaveAsButton() {
    if (saveAsButton == null) {
      saveAsButton = new JButton();
      saveAsButton.setIcon(new ImageIcon(getClass().getResource("SaveAs24.gif"))); //$NON-NLS-1$
      saveAsButton.setToolTipText(Messages.getString("MainFrame.SaveAs")); //$NON-NLS-1$
      saveAsButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          saveGroupAs();
        }
      });
    }
    return saveAsButton;
  }
  
  private void saveGroupAs() {
    GroupOperations.saveGroupAs(this);
    Preferences.userNodeForPackage(MainFrame.class).put("LastGroup_" + GameValues.getInstance().getName(), Group.getGroup().getFilePath()); //$NON-NLS-1$
  }

  /**
   * This method initializes settingsButton	
   * 	
   * @return javax.swing.JButton	
   */
  private JButton getSettingsButton() {
    if (settingsButton == null) {
      settingsButton = new JButton();
      settingsButton.setIcon(new ImageIcon(getClass().getResource("Preferences24.gif"))); //$NON-NLS-1$
      settingsButton.setToolTipText(Messages.getString("MainFrame.Settings")); //$NON-NLS-1$
      settingsButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          OptionsDialog dialog = new OptionsDialog(MainFrame.this);
          dialog.setModal(true);
          dialog.setVisible(true);
        }
      });
    }
    return settingsButton;
  }
  
  private JButton getAboutButton() {
    if (aboutButton == null) {
      aboutButton = new JButton();
      aboutButton.setIcon(new ImageIcon(getClass().getResource("About24.gif"))); //$NON-NLS-1$
      aboutButton.setToolTipText(Messages.getString("MainFrame.About")); //$NON-NLS-1$
      aboutButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          JOptionPane.showMessageDialog(MainFrame.this, Messages.getString("MainFrame.ProgramVersion") + Version.getCurrentVersionString() //$NON-NLS-1$
              + Messages.getString("MainFrame.ProgramAuthor"), "RPGameValues", JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
        }
      });
    }
    return aboutButton;
  }

  /* (non-Javadoc)
   * @see rpgamevalues.gui.FrameStateChanger#closeFrame(rpgamevalues.gui.SubFrame)
   */
  @Override
  public void closeFrame(SubFrame frame) {
    frame.dispose();
  }

  /* (non-Javadoc)
   * @see rpgamevalues.gui.FrameStateChanger#frameStateChanged(rpgamevalues.gui.SubFrame, boolean)
   */
  @Override
  public void frameStateChanged(SubFrame frame, boolean isOpen) {
  }

  /* (non-Javadoc)
   * @see rpgamevalues.gui.FrameStateChanger#openFrame(java.lang.String, java.awt.Rectangle)
   */
  @Override
  public void openFrame(String name, Rectangle bounds) {
    if (name.equals(getTitle())) {
      setBounds(bounds);
      return;
    }
    SubFrame.saveFrameBounds(name, bounds);
    HeroFrame frame = new HeroFrame(Group.getGroup().getHero(name));
    frame.setVisible(true);
  }

}  //  @jve:decl-index=0:visual-constraint="10,10"
