/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.gui;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.GroupLayout;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.GroupLayout.ParallelGroup;
import javax.swing.GroupLayout.SequentialGroup;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;

import rpgamevalues.control.Modificators;
import rpgamevalues.control.ValueCalculation;
import rpgamevalues.data.Hero;
import rpgamevalues.metadata.DiceSpecification;
import rpgamevalues.metadata.GameValues;
import rpgamevalues.metadata.IGameValue;


class ValueGroupPanel extends HeroChangerPanel {
  
  private String groupName;
  private Hero hero;
  
  private List<JLabel> labels = new ArrayList<JLabel>();
  private List<JComponent> currentValueFields = new ArrayList<JComponent>();
  private List<JTextField> defaultValueFields = new ArrayList<JTextField>();
  private List<JButton> add1Buttons = new ArrayList<JButton>();
  private List<JButton> add5Buttons = new ArrayList<JButton>();
  private List<JButton> remove1Buttons = new ArrayList<JButton>();
  private List<JButton> remove5Buttons = new ArrayList<JButton>();
  
  public ValueGroupPanel(IHeroFrame frame, Hero hero, String groupName) {
    super(frame);
    this.hero = hero;
    this.groupName = groupName;
    
    createLayout();
  }
  
  public void shownValuesToggled() {
    this.removeAll();
    createLayout();
  }
  
  private void createLayout() {
    
    boolean hasDefaultValues = false;
    
    labels.clear();
    currentValueFields.clear();
    defaultValueFields.clear();
    add1Buttons.clear();
    add5Buttons.clear();
    remove1Buttons.clear();
    remove5Buttons.clear();
    
    List<IGameValue> gameValues = GameValues.getInstance().getGameValues(groupName);
    for (IGameValue gameValue : gameValues) {
      if (hero.isValueShown(groupName, gameValue.getName(), gameValue.isVisibleByDefault())) {
        JLabel label = new JLabel(gameValue.getName() + ":"); //$NON-NLS-1$
        labels.add(label);
        JComponent textField = createCurrentValueField(gameValue);
        currentValueFields.add(textField);
        JTextField textField2 = createDefaultValueField(gameValue, textField);
        if (textField2 != null) {
          hasDefaultValues = true;
        }
        defaultValueFields.add(textField2);
        createButtons(gameValue, textField);
      }
    }
    
    JLabel nameLabel = new JLabel(Messages.getString("ValueGroupPanel.Value")); //$NON-NLS-1$
    JLabel defaultLabel = new JLabel(Messages.getString("ValueGroupPanel.Std")); //$NON-NLS-1$
    JLabel currentLabel = new JLabel(Messages.getString("ValueGroupPanel.Cur")); //$NON-NLS-1$
    
    GroupLayout layout = new GroupLayout(this);
    layout.setAutoCreateContainerGaps(true);
    layout.setAutoCreateGaps(true);
    
    ParallelGroup labelsGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    labelsGroup.addComponent(nameLabel);
    for (JLabel label : labels) {
      labelsGroup.addComponent(label);
    }
    ParallelGroup defaultValuesGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    if (hasDefaultValues) {
      defaultValuesGroup.addComponent(defaultLabel);
      for (JTextField field : defaultValueFields) {
        if (field != null) defaultValuesGroup.addComponent(field);
      }
    }
    ParallelGroup currentValuesGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    currentValuesGroup.addComponent(currentLabel);
    for (JComponent field : currentValueFields) {
      currentValuesGroup.addComponent(field);
    }
    ParallelGroup add1ButtonsGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    for (JButton button : add1Buttons) {
      if (button != null) add1ButtonsGroup.addComponent(button);
    }
    ParallelGroup add5ButtonsGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    for (JButton button : add5Buttons) {
      if (button != null) add5ButtonsGroup.addComponent(button);
    }
    ParallelGroup remove1ButtonsGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    for (JButton button : remove1Buttons) {
      if (button != null) remove1ButtonsGroup.addComponent(button);
    }
    ParallelGroup remove5ButtonsGroup = layout.createParallelGroup(GroupLayout.Alignment.LEADING);
    for (JButton button : remove5Buttons) {
      if (button != null) remove5ButtonsGroup.addComponent(button);
    }
    layout.setHorizontalGroup(layout.createSequentialGroup()
        .addGroup(labelsGroup)
        .addGroup(defaultValuesGroup)
        .addGroup(currentValuesGroup)
        .addGroup(add1ButtonsGroup)
        .addGroup(remove1ButtonsGroup)
        .addGroup(add5ButtonsGroup)
        .addGroup(remove5ButtonsGroup)
        );
    
    SequentialGroup verticalGroup = layout.createSequentialGroup();
    ParallelGroup labelsGroup2 = layout.createParallelGroup(GroupLayout.Alignment.BASELINE).addComponent(nameLabel);
    if (hasDefaultValues) {
      labelsGroup2 = labelsGroup2.addComponent(defaultLabel);
    }
    labelsGroup2 = labelsGroup2.addComponent(currentLabel);
    verticalGroup = verticalGroup.addGroup(labelsGroup2);
    for (int i = 0; i < currentValueFields.size(); ++i) {
      ParallelGroup valueGroup = layout.createParallelGroup(GroupLayout.Alignment.BASELINE);
      valueGroup.addComponent(labels.get(i));
      if (defaultValueFields.get(i) != null) {
        valueGroup.addComponent(defaultValueFields.get(i));
      }
      valueGroup.addComponent(currentValueFields.get(i));
      JButton add1Button = add1Buttons.get(i);
      if (add1Button != null) {
        valueGroup.addComponent(add1Button);
      }
      JButton remove1Button = remove1Buttons.get(i);
      if (remove1Button != null) {
        valueGroup.addComponent(remove1Button);
      }
      JButton add5Button = add5Buttons.get(i);
      if (add5Button != null) {
        valueGroup.addComponent(add5Button);
      }
      JButton remove5Button = remove5Buttons.get(i);
      if (remove5Button != null) {
        valueGroup.addComponent(remove5Button);
      }
      verticalGroup = verticalGroup.addGroup(valueGroup);
    }
    layout.setVerticalGroup(verticalGroup);
    
    setLayout(layout);
    
    heroChanged();
  }
  
  private class ModificationButtonListener implements ActionListener {
    
    private IGameValue gameValue;
    private JTextField currentValueField;
    private int change;
    
    public ModificationButtonListener(IGameValue gameValue, JTextField field, int change) {
      this.gameValue = gameValue;
      currentValueField = field;
      this.change = change;
    }

    public void actionPerformed(ActionEvent arg0) {
      int oldValue = ValueCalculation.<Integer>getCurrentValue(groupName, gameValue, hero);
      int newValue = oldValue + change;
      if (gameValue.hasMin() && newValue < gameValue.getMin()) 
        newValue = gameValue.getMin();
      if (gameValue.hasMax() && newValue > gameValue.getMax())
        newValue = gameValue.getMax();
      hero.setCurrentValue(groupName, gameValue.getName(), (Integer)newValue);
      listenForTextFields = false;
      listenForChanges = false;
      currentValueField.setText("" + Modificators.getModifiedValue(hero, groupName, gameValue)); //$NON-NLS-1$
      fireHeroChanged();
      listenForChanges = true;
      listenForTextFields = true;
    }
    
  }
  
  private class CheckBoxListener implements ActionListener {
    private IGameValue gameValue;
    
    public CheckBoxListener(IGameValue gameValue) {
      this.gameValue = gameValue;
    }
    
    public void actionPerformed(ActionEvent arg0) {
      boolean oldValue = ValueCalculation.<Boolean>getCurrentValue(groupName, gameValue, hero);
      hero.setCurrentValue(groupName, gameValue.getName(), !oldValue);
      fireHeroChanged();
    }
  }
  
  private class DiceListener implements ActionListener {
    private IGameValue gameValue;
    
    public DiceListener(IGameValue gameValue) {
      this.gameValue = gameValue;
    }
    
    public void actionPerformed(ActionEvent e) {
      DiceSpecification ds = hero.getCurrentValue(groupName, gameValue.getName());
      int result = ds.calcValue();
      JOptionPane.showMessageDialog(ValueGroupPanel.this, Messages.getString("ValueGroupPanel.DiceRollResult") + result, "RPGameValues", JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$ //$NON-NLS-2$
    }
  }
  
  private static Icon sDiceIcon = null;
  
  private static Icon getDiceIcon() {
    if (sDiceIcon == null) { 
      sDiceIcon = new ImageIcon(ValueGroupPanel.class.getResource("probe.png")); //$NON-NLS-1$
    }
    return sDiceIcon;
  }

  /**
   * @param gameValue
   */
  private void createButtons(IGameValue gameValue, JComponent currentValueField) {
    if (gameValue.isCalculated()) {
      add1Buttons.add(null);
      add5Buttons.add(null);
      remove1Buttons.add(null);
      remove5Buttons.add(null);
      return;
    }
    switch (gameValue.getGameValueType()) {
    case Boolean:
    case String:
      {
        add1Buttons.add(null);
        add5Buttons.add(null);
        remove1Buttons.add(null);
        remove5Buttons.add(null);
        return;
      }
    case DiceSpecification:
      {
        JButton diceButton = new JButton();
        diceButton.setIcon(getDiceIcon());
        diceButton.addActionListener(new DiceListener(gameValue));
        add1Buttons.add(diceButton);
        add5Buttons.add(null);
        remove1Buttons.add(null);
        remove5Buttons.add(null);
        break;
      }
    default:
      {
        JTextField cvField = (JTextField)currentValueField;
        JButton add1Button = new JButton("+1"); //$NON-NLS-1$
        add1Button.addActionListener(new ModificationButtonListener(gameValue, cvField, 1));
        add1Buttons.add(add1Button);
        JButton add5Button = new JButton("+5"); //$NON-NLS-1$
        add5Button.addActionListener(new ModificationButtonListener(gameValue, cvField, 5));
        add5Buttons.add(add5Button);
        JButton rem1Button = new JButton("-1"); //$NON-NLS-1$
        rem1Button.addActionListener(new ModificationButtonListener(gameValue, cvField, -1));
        remove1Buttons.add(rem1Button);
        JButton remove5Button = new JButton("-5"); //$NON-NLS-1$
        remove5Button.addActionListener(new ModificationButtonListener(gameValue, cvField, -5));
        remove5Buttons.add(remove5Button);
        break;
      }
    }
  }
  
  private boolean listenForTextFields = true;
  private boolean listenForChanges = true;
  
  private abstract class GameValueDocumentListener implements DocumentListener {
    
    private IGameValue gameValue;
    
    protected JTextField textField;
    
    protected GameValueDocumentListener(IGameValue gameValue, JTextField textField) {
      this.gameValue = gameValue;
      this.textField = textField;
    }

    /* (non-Javadoc)
     * @see javax.swing.event.DocumentListener#changedUpdate(javax.swing.event.DocumentEvent)
     */
    @Override
    public void changedUpdate(DocumentEvent arg0) {
      changed(arg0);
    }

    /* (non-Javadoc)
     * @see javax.swing.event.DocumentListener#insertUpdate(javax.swing.event.DocumentEvent)
     */
    @Override
    public void insertUpdate(DocumentEvent arg0) {
      changed(arg0);
    }

    /* (non-Javadoc)
     * @see javax.swing.event.DocumentListener#removeUpdate(javax.swing.event.DocumentEvent)
     */
    @Override
    public void removeUpdate(DocumentEvent arg0) {
      changed(arg0);
    }
    
    private void changed(DocumentEvent docEvent) {
      if (!listenForTextFields) {
        return;
      }
      boolean changed = false;
      switch (gameValue.getGameValueType()) {
      case DiceSpecification:
        changed = changedDS(docEvent);
        break;
      case String:
    	changed = changedString(docEvent);
    	break;
      default:
        changed = changedNumber(docEvent);
        break;
      }
      if (changed) {
        listenForChanges = false;
        fireHeroChanged();
        listenForChanges = true;
      }
    }
    
    private boolean changedNumber(DocumentEvent docEvent) {
      try {
        int value = Integer.parseInt(docEvent.getDocument().getText(0, docEvent.getDocument().getLength()));
        if (gameValue.hasMax() && value > gameValue.getMax()) {
          textField.setForeground(Color.RED);
          textField.setToolTipText(String.format(Messages.getString("ValueGroupPanel.ValueTooHigh"), gameValue.getMax())); //$NON-NLS-1$
          return false;
        }
        if (gameValue.hasMin() && value < gameValue.getMin()) {
          textField.setForeground(Color.RED);
          textField.setToolTipText(String.format(Messages.getString("ValueGroupPanel.ValueTooLow"), gameValue.getMax())); //$NON-NLS-1$
          return false;
        }
        textField.setForeground(Color.BLACK);
        textField.setToolTipText(""); //$NON-NLS-1$
        changed(gameValue, (Integer)value);
        return true;
      }
      catch (NumberFormatException e) {
        textField.setForeground(Color.RED);
        textField.setToolTipText(Messages.getString("ValueGroupPanel.ValueMustBeNumber")); //$NON-NLS-1$
        return false;
      }
      catch (BadLocationException e) {
        textField.setForeground(Color.RED);
        textField.setToolTipText(Messages.getString("ValueGroupPanel.ValueMissing")); //$NON-NLS-1$
        return false;
      }
    }
    
    private boolean changedDS(DocumentEvent docEvent) {
      try {
        DiceSpecification value = DiceSpecification.parse(docEvent.getDocument().getText(0, docEvent.getDocument().getLength()));
        textField.setForeground(Color.BLACK);
        textField.setToolTipText(""); //$NON-NLS-1$
        changed(gameValue, value);
        return true;
      }
      catch (NumberFormatException e) {
        textField.setForeground(Color.RED);
        textField.setToolTipText(Messages.getString("ValueGroupPanel.ValueMustBeDiceSpecification")); //$NON-NLS-1$
        return false;
      }
      catch (BadLocationException e) {
        textField.setForeground(Color.RED);
        textField.setToolTipText(Messages.getString("ValueGroupPanel.ValueMissing")); //$NON-NLS-1$
        return false;
      }
    }
    
    private boolean changedString(DocumentEvent docEvent) {
      try {
    	String value = docEvent.getDocument().getText(0, docEvent.getDocument().getLength());
    	changed(gameValue, value);
    	return true;
      }
      catch (BadLocationException e) {
    	changed(gameValue, "");
    	return true;
      }
    }
    
    protected abstract <T> void changed(IGameValue gameValue, T value);
  }
  
  private class CurrentValueListener extends GameValueDocumentListener {
    public CurrentValueListener(IGameValue gameValue, JTextField textField) {
      super(gameValue, textField);
    }
    
    @SuppressWarnings("unchecked")
	protected <T> void changed(IGameValue gameValue, T value) {
      if (gameValue.getGameValueType() == IGameValue.GameValueType.Number) {
        T oldValue = ValueCalculation.<T>getCurrentValue(groupName, gameValue, hero);
    	int modifiedValue = Modificators.getModifiedValue(hero, groupName, gameValue);
    	value = (T)(Integer)((Integer)value - modifiedValue + (Integer)oldValue);
      }
      hero.setCurrentValue(groupName, gameValue.getName(), value);
    }
  }

  private class DefaultValueListener extends GameValueDocumentListener {
    private JTextField currentValueField;
    
    public DefaultValueListener(IGameValue gameValue, JTextField textField, JTextField currentValueField) {
      super(gameValue, textField);
      this.currentValueField = currentValueField;
    }
    
    protected <T> void changed(IGameValue gameValue, T value) {
      hero.setDefaultValue(groupName, gameValue.getName(), value);
      hero.setCurrentValue(groupName, gameValue.getName(), value);
      if (gameValue.getGameValueType() == IGameValue.GameValueType.Number) {
        currentValueField.setText("" + Modificators.getModifiedValue(hero, groupName, gameValue)); //$NON-NLS-1$
      }
      else {
        DiceSpecification ds = ValueCalculation.<DiceSpecification>getCurrentValue(groupName, gameValue, hero);
        currentValueField.setText(ds.toString());
      }
    }
  }

  /**
   * @param gameValue
   * @return
   */
  private JTextField createDefaultValueField(IGameValue gameValue, JComponent currentValueField) {
    if (gameValue.hasDefault()) {
      JTextField textField = new JTextField();
      textField.getDocument().addDocumentListener(new DefaultValueListener(gameValue, textField, (JTextField)currentValueField));
      if (gameValue.isCalculated()) {
        textField.setEditable(false);
      }
      return textField;
    }
    else {
      return null;
    }
  }

  /**
   * @param name
   * @return
   */
  private JComponent createCurrentValueField(IGameValue gameValue) {
    switch (gameValue.getGameValueType()) {
    case Boolean:
    {
      JCheckBox checkBox = new JCheckBox();
      checkBox.addActionListener(new CheckBoxListener(gameValue));
      if (gameValue.isCalculated()) {
        checkBox.setEnabled(false);
      }
      return checkBox;
    }
    default:
    {
      JTextField textField = new JTextField();
      textField.getDocument().addDocumentListener(new CurrentValueListener(gameValue, textField));
      if (gameValue.isCalculated()) {
        textField.setEditable(false);
      }
      return textField;
    }
    }
  }
  
  public void heroChanged() {
    if (!listenForChanges) return;
    listenForTextFields = false;
    int i = 0;
    for (IGameValue gameValue : GameValues.getInstance().getGameValues(groupName)) {
      if (hero.isValueShown(groupName, gameValue.getName(), gameValue.isVisibleByDefault())) {
    	switch (gameValue.getGameValueType()) {
    	case Number:
          ((JTextField)currentValueFields.get(i)).setText("" + Modificators.getModifiedValue(hero, groupName, gameValue)); //$NON-NLS-1$
          break;
    	case Boolean:
          ((JCheckBox)currentValueFields.get(i)).setSelected(ValueCalculation.<Boolean>getCurrentValue(groupName, gameValue, hero));
          break;
    	case String:
    	  ((JTextField)currentValueFields.get(i)).setText(ValueCalculation.<String>getCurrentValue(groupName, gameValue, hero));
    	  break;
    	case DiceSpecification:
          DiceSpecification ds = ValueCalculation.<DiceSpecification>getCurrentValue(groupName, gameValue, hero);
          ((JTextField)currentValueFields.get(i)).setText(ds.toString());
          break;
        default:
          break;
    	}
        if (gameValue.hasDefault()) {
          defaultValueFields.get(i).setText("" + ValueCalculation.getDefaultValue(groupName, gameValue, hero)); //$NON-NLS-1$
        }
        ++i;
      }
    }
    listenForTextFields = true;
  }

}
