/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.metadata;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public final class GameValues {
	
  private String name;
  
  public String getName() {
	  return name;
  }

  public List<String> getGroups() {
    return mGroupNames;
  }
  
  public List<IGameValue> getGameValues(String group) {
    if (mValues.containsKey(group)) {
      return mValues.get(group);
    }
    else {
      return new ArrayList<IGameValue>();
    }
  }
  
  public boolean hasConfigurations(String groupName) {
    if (mGroups.containsKey(groupName)) {
      return mGroups.get(groupName).hasMultipleConfigurations();
    }
    else {
      return false;
    }
  }
  
  public void readFile(String fileName) throws IOException {
    try {
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
      DocumentBuilder db = dbf.newDocumentBuilder();
      Document dom = db.parse(fileName);

      Element docEle = dom.getDocumentElement();
      
      name = docEle.getAttribute("name");
      if (name == null) {
    	  throw new IOException("Missing name for Metadata");
      }
      
      NodeList nl = docEle.getElementsByTagName("group"); //$NON-NLS-1$
      for(int i = 0 ; i < nl.getLength();i++) {
        Element el = (Element)nl.item(i);
        GameValueGroup group = readGroup(el);
        mGroups.put(group.getName(), group);
        mGroupNames.add(group.getName());
      }
    }
    catch(ParserConfigurationException pce) {
      throw new IOException(pce.getLocalizedMessage(), pce);
    }
    catch(SAXException se) {
      throw new IOException(se.getLocalizedMessage(), se);
    }
    catch (IOException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
  }
  
  public static GameValues getInstance() {
    if (sInstance == null) {
      sInstance = new GameValues();
    }
    return sInstance;
  }
  
  private GameValueGroup readGroup(Element el) throws IOException {
    String groupName = el.getAttribute("name"); //$NON-NLS-1$
    if (groupName == null) {
      throw new IOException(Messages.getString("GameValues.GameValueGroupNameMissing")); //$NON-NLS-1$
    }
    if (mGroups.containsKey(groupName)) {
      throw new IOException(Messages.getString("GameValues.GameValueGroupDuplicate") + groupName); //$NON-NLS-1$
    }
    boolean configs = Boolean.parseBoolean(el.getAttribute("hasConfigurations")); //$NON-NLS-1$
    if (!mValues.containsKey(groupName)) {
      mValues.put(groupName, new ArrayList<IGameValue>());
    }
    NodeList nl = el.getElementsByTagName("gamevalue"); //$NON-NLS-1$
    for (int i = 0; i < nl.getLength(); ++i) {
      Element el2 = (Element)nl.item(i);
      IGameValue gameValue = readGameValue(el2);
      if (gameValue == null) continue;
      if (gameValue.getName() == null) {
        throw new IOException(Messages.getString("GameValues.GameValueNameMissing")); //$NON-NLS-1$
      }
      mValues.get(groupName).add(gameValue);      
    }
    return new GameValueGroup(groupName, configs);
  }
  
  private IGameValue readGameValue(Element el) throws IOException {
    String className = el.getAttribute("class"); //$NON-NLS-1$
    if (className == null) {
      throw new IOException(Messages.getString("GameValues.GameValueClassNameMissing")); //$NON-NLS-1$
    }
    try {
      Class<?> theClass = Class.forName(className);
      Object theObject = theClass.newInstance();
      IGameValue gameValue = (IGameValue)theObject;

      NodeList nl = el.getElementsByTagName("property"); //$NON-NLS-1$
      if (nl.getLength() != 0) {
        Method m = theClass.getMethod("setProperty", String.class, String.class); //$NON-NLS-1$
        for(int i = 0 ; i < nl.getLength();i++) {
          Element ele = (Element)nl.item(i);
          String propertyName = ele.getAttribute("name"); //$NON-NLS-1$
          if (propertyName == null) {
            throw new IOException(Messages.getString("GameValues.PropertyNameMissing")); //$NON-NLS-1$
          }
          String propertyValue = ele.getAttribute("value"); //$NON-NLS-1$
          if (propertyValue == null) {
            throw new IOException(Messages.getString("GameValues.PropertyValueMissing")); //$NON-NLS-1$
          }
          m.invoke(theObject, propertyName, propertyValue);
        }        
      }
      
      return gameValue;
    }
    catch (ClassNotFoundException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (IllegalAccessException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (InstantiationException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (ClassCastException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (NoSuchMethodException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (InvocationTargetException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (NumberFormatException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
    catch (IllegalArgumentException e) {
      throw new IOException(e.getLocalizedMessage(), e);
    }
  }
  
  private GameValues() {
    mGroupNames = new ArrayList<String>();
    mValues = new HashMap<String, ArrayList<IGameValue>>();
    mGroups = new HashMap<String, GameValueGroup>();
  }
  
  private HashMap<String, ArrayList<IGameValue>> mValues;
  private ArrayList<String> mGroupNames;
  private HashMap<String, GameValueGroup> mGroups;
  
  private static GameValues sInstance = null;
}
