/*
 Copyright (c) 2006-2009 [Joerg Ruedenauer]
 
 This file is part of RPGameValues.

 RPGameValues is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 RPGameValues is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with RPGameValues; if not, write to the Free Software
 Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package rpgamevalues.metadata.logic;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import rpgamevalues.metadata.Hero;

public abstract class PropertyBasedValue extends AbstractCalculatedValue {

  private int getSafeInt(Integer value) {
    return value != null ? value : 12;
  }
  
  /* (non-Javadoc)
   * @see rpgamevalues.metadata.IGameValue#calcCurrentValue(rpgamevalues.metadata.Hero)
   */
  @Override
  public final int calcCurrentValue(Hero hero) {
    int sum = 0;
    List<String> names = new ArrayList<String>();
    List<String> groups = new ArrayList<String>();
    getProperties(hero, groups, names);
    for (int i = 0 ; i < groups.size(); ++i) {
      sum += getSafeInt(hero.<Integer>getCurrentValue(groups.get(i), names.get(i)));
    }
    return (int)Math.round((float)sum / (float)divisor);
  }

  /* (non-Javadoc)
   * @see rpgamevalues.metadata.IGameValue#calcDefaultValue(rpgamevalues.metadata.Hero)
   */
  @Override
  public final int calcDefaultValue(Hero hero) {
    int sum = 0;
    List<String> names = new ArrayList<String>();
    List<String> groups = new ArrayList<String>();
    getProperties(hero, groups, names);
    for (int i = 0 ; i < groups.size(); ++i) {
      sum += getSafeInt(hero.<Integer>getDefaultValue(groups.get(i), names.get(i)));
    }
    return (int)Math.round((float)sum / (float)divisor);
  }
  
  protected void getProperties(Hero hero, List<String> groups, List<String> names) {
    groups.addAll(propertyGroups);
    names.addAll(propertyNames);
  }
  
  public final String getName() {
    return name;
  }
  
  protected PropertyBasedValue() {
    divisor = 1;
    name = ""; //$NON-NLS-1$
    propertyNames = new ArrayList<String>();
    propertyGroups = new ArrayList<String>();
    affectedValues = new HashMap<String, Set<String>>();
    visible = true;
  }
  
  protected final void setName(String name) {
    this.name = name;
  }
  
  protected final void setDivisor(int divisor) {
    this.divisor = divisor;
  }
  
  protected final void addProperty(String property) {
	StringTokenizer tokenizer = new StringTokenizer(property, "/");
	if (tokenizer.countTokens() != 2) {
	  throw new IllegalArgumentException("Format must be GroupName/ValueName");
	}
	propertyGroups.add(tokenizer.nextToken());
    propertyNames.add(tokenizer.nextToken());
  }
  
  protected final void addAffectedValue(String value) {
	StringTokenizer tokenizer = new StringTokenizer(value, "/");
	if (tokenizer.countTokens() != 2) {
	  throw new IllegalArgumentException("Format must be GroupName/ValueName");
	}
	String group = tokenizer.nextToken();
	String name = tokenizer.nextToken();
	if (!affectedValues.containsKey(group)) {
		affectedValues.put(group, new HashSet<String>());
	}
    affectedValues.get(group).add(name);
  }
  
  protected final void setVisible(boolean visible) {
    this.visible = visible;
  }
  
  public final int getValueModificator(int currentValue, int defaultValue, String groupName,
      String valueName) {
    if (affectedValues.containsKey(groupName) && affectedValues.get(groupName).contains(valueName)) {
      return currentValue - defaultValue;
    }
    else {
      return 0;
    }
  }
  
  public final boolean isVisibleByDefault() {
    return visible;
  }
  
  private int divisor;
  private List<String> propertyNames;
  private List<String> propertyGroups;
  private String name;
  private Map<String, Set<String>> affectedValues;
  private boolean visible;

}
